/*
** file: algoritmos.cs
**
** IAResgate
** UC: 21071 - IIA @ UAb
** e-fólio A 2024-25
**
** Aluno: 2300163 - Luis Pereira
*/

namespace IAResgate
{
    // Classe que representa um estado do agente no ambiente durante a busca
    public class Estado
    {
        public (int x, int y) Posicao { get; }                       // Coordenadas do agente no mapa
        public int TempoRestante { get; }                            // Tempo restante para concluir a missão
        public int VisitantesResgatados { get; }                     // Número de visitantes já resgatados
        public HashSet<(int, int)> VisitantesVisitados { get; }     // Conjunto de visitantes já visitados
        public List<(int, int)> Caminho { get; }                     // Caminho percorrido desde a entrada
        public int CustoTotal { get; }                               // Custo acumulado (tempo gasto)

        // Construtor do estado
        public Estado(
            (int, int) posicao,
            int tempo,
            int resgatados,
            HashSet<(int, int)> visitados,
            List<(int, int)> caminho,
            int custoTotal
        )
        {
            Posicao = posicao;
            TempoRestante = tempo;
            VisitantesResgatados = resgatados;
            VisitantesVisitados = new HashSet<(int, int)>(visitados);  // Clone para evitar referência externa
            Caminho = new List<(int, int)>(caminho);                   // Clone do caminho até este ponto
            CustoTotal = custoTotal;
        }
    }

    // Resultado base retornado
    public class ResultadoUCS
    {
        public List<(int, int)> Caminho { get; }         // Caminho final seguido
        public int TempoRestante { get; }                // Tempo restante ao concluir
        public int CustoTotal { get; }                   // Custo acumulado final
        public int VisitantesResgatados { get; }         // Quantos visitantes foram resgatados

        public ResultadoUCS(
            List<(int, int)> caminho,
            int tempoRestante,
            int custoTotal,
            int resgatados
        )
        {
            Caminho = caminho;
            TempoRestante = tempoRestante;
            CustoTotal = custoTotal;
            VisitantesResgatados = resgatados;
        }
    }

    // Resultado da busca em profundidade
    public class ResultadoDFS : ResultadoUCS //Herda de ResultadoUCS
    {
        public ResultadoDFS(
            List<(int, int)> caminho,
            int tempoRestante,
            int custoTotal,
            int resgatados
        )
            : base(caminho, tempoRestante, custoTotal, resgatados) { }
    }

    // Resultado da busca em largura
    public class ResultadoBFS : ResultadoUCS //Herda de ResultadoUCS
    {
        public ResultadoBFS(
            List<(int, int)> caminho,
            int tempoRestante,
            int custoTotal,
            int resgatados
        )
            : base(caminho, tempoRestante, custoTotal, resgatados) { }
    }

    // Classe que contém as implementações dos algoritmos de busca
    public static class Algoritmos
    {
        // Movimentos possíveis no mapa: cima, direita, baixo, esquerda
        private static readonly (int dx, int dy)[] Movimentos =
        {
            (0, 1),   // Cima
            (1, 0),   // Direita
            (0, -1),  // Baixo
            (-1, 0)   // Esquerda
        };

        // Implementação do algoritmo UCS (Uniform Cost Search)
        public static bool UCS(
            Instancia instancia,
            out ResultadoUCS resultado,
            ref int expansoes,
            ref int geracoes
        )
        {
            // Dados iniciais da instância
            var mapa = instancia.Mapa;
            int n = instancia.N;
            int k = instancia.K;
            int tempo = instancia.Tempo;
            int meio = n / 2;
            
            // Inicialização do conjunto de visitantes
            var visitantesTotais = new HashSet<(int, int)>();
            for (int x = 0; x < n; x++)
            {
                for (int y = 0; y < n; y++)
                {
                    if (mapa[x, y] < 0) // é visitante
                        visitantesTotais.Add((x, y));
                }
            }

            // Definição das quatro portas de entrada/saída
            var portas = new List<(int, int)>
            {
                (0, meio),       // Topo
                (n - 1, meio),   // Base
                (meio, 0),       // Esquerda
                (meio, n - 1)    // Direita
            };

            // Fila de prioridade para UCS (min-heap baseada no custo acumulado)
            var fila = new PriorityQueue<Estado, int>();
            var visitados = new HashSet<string>(); // Estados já processados

            // Inicializa a fila com os estados iniciais (partida de cada porta)
            foreach (var entrada in portas)
            {
                int custoEntrada = mapa[entrada.Item1, entrada.Item2];
                int tempoGasto = custoEntrada > 0 ? custoEntrada : 1; // mínimo 1
                int tempoDepois = tempo - tempoGasto;

                if (tempoDepois < 0) continue; // Ignora se não há tempo suficiente

                var estadoInicial = new Estado(
                    entrada,
                    tempoDepois,
                    0,
                    new HashSet<(int, int)>(),
                    new List<(int, int)> { entrada },
                    tempoGasto
                );

                fila.Enqueue(estadoInicial, tempoGasto);
                geracoes++;
            }

            // Loop principal da busca
            while (fila.Count > 0)
            {
                var estado = fila.Dequeue();
                expansoes++;

                // Verifica se está numa porta de saída e já resgatou k visitantes
                if (
                    portas.Contains(estado.Posicao) &&
                    estado.VisitantesResgatados >= k &&
                    estado.TempoRestante >= 1
                )
                {
                    int custoSaida = 1;

                    // Verifica se ainda há tempo para sair
                    if (estado.TempoRestante >= custoSaida)
                    {
                        resultado = new ResultadoUCS(
                            new List<(int, int)>(estado.Caminho),
                            estado.TempoRestante - custoSaida,
                            estado.CustoTotal + custoSaida,
                            estado.VisitantesResgatados
                        );

                        return true;
                    }
                }

                // Evita expandir estados repetidos
                string chave = EstadoToChave(estado);
                if (visitados.Contains(chave)) continue;
                visitados.Add(chave);

                // Geração dos estados sucessores
                foreach (var (dx, dy) in Movimentos)
                {
                    int nx = estado.Posicao.x + dx;
                    int ny = estado.Posicao.y + dy;

                    // Verifica limites do mapa
                    if (nx < 0 || ny < 0 || nx >= n || ny >= n) continue;

                    int custo = mapa[nx, ny];
                    if (custo == 10) continue; // obstáculo intransponível

                    int tempoGasto = custo > 0 ? custo : 1;
                    int tempoNovo = estado.TempoRestante - tempoGasto;
                    if (tempoNovo < 0) continue;

                    int novosResgatados = estado.VisitantesResgatados;
                    var novosVisitados = new HashSet<(int, int)>(estado.VisitantesVisitados);

                    // Se for um visitante ainda não visitado, ganhamos tempo
                    if (custo < 0 && !novosVisitados.Contains((nx, ny)))
                    {
                        tempoNovo += Math.Abs(custo);
                        novosResgatados++;
                        novosVisitados.Add((nx, ny));
                    }
                    // impossível atingir K visitantes com os restantes disponíveis
                    int aindaPossiveis = visitantesTotais.Count - novosVisitados.Count;
                    if (novosResgatados + aindaPossiveis < k) continue;


                    int novoCustoTotal = estado.CustoTotal + tempoGasto;
                    var novoCaminho = new List<(int, int)>(estado.Caminho) { (nx, ny) };

                    var novoEstado = new Estado(
                        (nx, ny),
                        tempoNovo,
                        novosResgatados,
                        novosVisitados,
                        novoCaminho,
                        novoCustoTotal
                    );

                    fila.Enqueue(novoEstado, novoCustoTotal);
                    geracoes++;
                }
            }

            resultado = null!;
            return false;
        }

        // Implementação do algoritmo DFS (Depth-First Search)
        public static bool DFS(
            Instancia instancia,
            out ResultadoDFS resultado,
            ref int expansoes,
            ref int geracoes
        )
        {
            // Dados iniciais da instância
            var mapa = instancia.Mapa;
            int n = instancia.N;
            int k = instancia.K;
            int tempo = instancia.Tempo;
            int meio = n / 2;
            
            // Inicialização do conjunto de visitantes
            var visitantesTotais = new HashSet<(int, int)>();
            for (int x = 0; x < n; x++)
            {
                for (int y = 0; y < n; y++)
                {
                    if (mapa[x, y] < 0) // é visitante
                        visitantesTotais.Add((x, y));
                }
            }

            // Portas de entrada/saída
            var portas = new List<(int, int)>
            {
                (0, meio),
                (n - 1, meio),
                (meio, 0),
                (meio, n - 1)
            };

            var pilha = new Stack<Estado>();                // Pilha para DFS
            var visitados = new HashSet<string>();          // Estados já visitados

            // Inicializar com estados de cada porta
            foreach (var entrada in portas)
            {
                int custoEntrada = mapa[entrada.Item1, entrada.Item2];
                int tempoGasto = custoEntrada > 0 ? custoEntrada : 1;
                int tempoDepois = tempo - tempoGasto;

                if (tempoDepois < 0) continue;

                pilha.Push(
                    new Estado(
                        entrada,
                        tempoDepois,
                        0,
                        new HashSet<(int, int)>(),
                        new List<(int, int)> { entrada },
                        tempoGasto
                    )
                );

                geracoes++;
            }

            // Loop principal
            while (pilha.Count > 0)
            {
                var estado = pilha.Pop();
                expansoes++;

                // Verifica se é solução válida
                if (
                    portas.Contains(estado.Posicao) &&
                    estado.VisitantesResgatados >= k &&
                    estado.TempoRestante >= 1
                )
                {
                    int custoSaida = 1;

                    if (estado.TempoRestante >= custoSaida)
                    {
                        resultado = new ResultadoDFS(
                            new List<(int, int)>(estado.Caminho),
                            estado.TempoRestante - custoSaida,
                            estado.CustoTotal + custoSaida,
                            estado.VisitantesResgatados
                        );
                        return true;
                    }
                }

                // Evitar repetir estados
                string chave = EstadoToChave(estado);
                if (visitados.Contains(chave)) continue;
                visitados.Add(chave);

                // Gerar novos estados
                foreach (var (dx, dy) in Movimentos)
                {
                    int nx = estado.Posicao.x + dx;
                    int ny = estado.Posicao.y + dy;

                    if (nx < 0 || ny < 0 || nx >= n || ny >= n) continue;

                    int valor = mapa[nx, ny];
                    if (valor == 10) continue;

                    int custo = valor > 0 ? valor : 1;
                    int tempoNovo = estado.TempoRestante - custo;
                    if (tempoNovo < 0) continue;

                    int novosResgatados = estado.VisitantesResgatados;
                    var novosVisitados = new HashSet<(int, int)>(estado.VisitantesVisitados);

                    if (valor < 0 && !novosVisitados.Contains((nx, ny)))
                    {
                        tempoNovo += Math.Abs(valor);
                        novosResgatados++;
                        novosVisitados.Add((nx, ny));
                    }
                    // impossível atingir K visitantes com os restantes disponíveis
                    int aindaPossiveis = visitantesTotais.Count - novosVisitados.Count;
                    if (novosResgatados + aindaPossiveis < k) continue;
                    
                    var novoCaminho = new List<(int, int)>(estado.Caminho) { (nx, ny) };

                    var novoEstado = new Estado(
                        (nx, ny),
                        tempoNovo,
                        novosResgatados,
                        novosVisitados,
                        novoCaminho,
                        estado.CustoTotal + custo
                    );
                    pilha.Push(novoEstado);
                    geracoes++;
                }
            }
            resultado = null!;
            return false;
        }

        // Implementação do algoritmo BFS (Breadth-First Search)
        public static bool BFS(
            Instancia instancia,
            out ResultadoBFS resultado,
            ref int expansoes,
            ref int geracoes
        )
        {
            // Dados iniciais da instância
            var mapa = instancia.Mapa;
            int n = instancia.N;
            int k = instancia.K;
            int tempo = instancia.Tempo;
            int meio = n / 2;
            
            // Inicialização do conjunto de visitantes
            var visitantesTotais = new HashSet<(int, int)>();
            for (int x = 0; x < n; x++)
            {
                for (int y = 0; y < n; y++)
                {
                    if (mapa[x, y] < 0) // é visitante
                        visitantesTotais.Add((x, y));
                }
            }

            // Portas de entrada/saída
            var portas = new List<(int, int)>
            {
                (0, meio),
                (n - 1, meio),
                (meio, 0),
                (meio, n - 1)
            };

            var fila = new Queue<Estado>();                 // Fila para BFS
            var visitados = new HashSet<string>();          // Estados já visitados
            

            // Inicialização
            foreach (var entrada in portas)
            {
                int custoEntrada = mapa[entrada.Item1, entrada.Item2];
                int tempoGasto = custoEntrada > 0 ? custoEntrada : 1;
                int tempoDepois = tempo - tempoGasto;

                if (tempoDepois < 0) continue;

                fila.Enqueue(
                    new Estado(
                        entrada,
                        tempoDepois,
                        0,
                        new HashSet<(int, int)>(),
                        new List<(int, int)> { entrada },
                        tempoGasto
                    )
                );
                geracoes++;
            }
            // Loop principal
            while (fila.Count > 0)
            {
                var estado = fila.Dequeue();
                expansoes++;

                // Verifica se é uma solução válida
                if (
                    portas.Contains(estado.Posicao) &&
                    estado.VisitantesResgatados >= k &&
                    estado.TempoRestante >= 1
                )
                {
                    int custoSaida = 1;

                    if (estado.TempoRestante >= custoSaida)
                    {
                        resultado = new ResultadoBFS(
                            new List<(int, int)>(estado.Caminho),
                            estado.TempoRestante - custoSaida,
                            estado.CustoTotal + custoSaida,
                            estado.VisitantesResgatados
                        );
                        return true;
                    }
                }

                // Verifica se já visitámos este estado
                string chave = EstadoToChave(estado);
                if (visitados.Contains(chave)) continue;
                visitados.Add(chave);

                // Geração de novos estados
                foreach (var (dx, dy) in Movimentos)
                {
                    int nx = estado.Posicao.x + dx;
                    int ny = estado.Posicao.y + dy;

                    if (nx < 0 || ny < 0 || nx >= n || ny >= n) continue;

                    int valor = mapa[nx, ny];
                    if (valor == 10) continue;

                    int custo = valor > 0 ? valor : 1;
                    int tempoNovo = estado.TempoRestante - custo;
                    if (tempoNovo < 0) continue;

                    int novosResgatados = estado.VisitantesResgatados;
                    var novosVisitados = new HashSet<(int, int)>(estado.VisitantesVisitados);

                    if (valor < 0 && !novosVisitados.Contains((nx, ny)))
                    {
                        tempoNovo += Math.Abs(valor);
                        novosResgatados++;
                        novosVisitados.Add((nx, ny));
                    }
                    // impossível atingir K visitantes com os restantes disponíveis
                    int aindaPossiveis = visitantesTotais.Count - novosVisitados.Count;
                    if (novosResgatados + aindaPossiveis < k) continue;

                    var novoCaminho = new List<(int, int)>(estado.Caminho) { (nx, ny) };

                    var novoEstado = new Estado(
                        (nx, ny),
                        tempoNovo,
                        novosResgatados,
                        novosVisitados,
                        novoCaminho,
                        estado.CustoTotal + custo
                    );

                    fila.Enqueue(novoEstado);
                    geracoes++;
                }
            }

            resultado = null!;
            return false;
        }

        // Função utilitária: gera uma string única para representar um estado
        private static string EstadoToChave(Estado estado)
        {
            string visitantes = string.Join(";", estado.VisitantesVisitados
                .OrderBy(v => v.Item1)
                .ThenBy(v => v.Item2)
                .Select(v => $"{v.Item1},{v.Item2}"));

            return $"{estado.Posicao.x},{estado.Posicao.y},{estado.TempoRestante},{estado.VisitantesResgatados}:{visitantes}";
        }
    }
}