/*
** file: algoritmos.cs
**
** IAResgateB
** UC: 21071 - IIA @ UAb
** e-fólio B 2024-25
**
** Aluno: 2300163 - Luis Pereira
*/

namespace IAResgateB
{
    // Representa o estado de um ponto do percurso durante a execução do algoritmo
    public class EstadoPasseio
    {
        public (int, int) Posicao { get; }  // Posição atual
        public int TempoRestante { get; }   // Tempo disponível
        public int Satisfacao { get; }      // Pontuação de satisfação
        public int PenalizacaoAtual { get; } // Penalizações por repetição
        public HashSet<(int, int)> Visitados { get; } // Casas visitadas
        public List<(int, int)> Caminho { get; }      // Caminho percorrido
        public int Heuristica { get; }      // Heurística estimada

        // Construtor do estado
        public EstadoPasseio(
            (int, int) posicao,
            int tempoRestante,
            int satisfacao,
            int penalizacaoAtual,
            HashSet<(int, int)> visitados,
            List<(int, int)> caminho,
            int heuristica)
        {
            Posicao = posicao;
            TempoRestante = tempoRestante;
            Satisfacao = satisfacao;
            PenalizacaoAtual = penalizacaoAtual;
            // Cria cópia do conjunto de visitados
            Visitados = new HashSet<(int, int)>(visitados);
            // Cria cópia do caminho percorrido
            Caminho = new List<(int, int)>(caminho);
            Heuristica = heuristica;
        }
    }
    public static class Algoritmos
    {
        // Contadores globais
        // Quantidade de estados explorados (retirados da fila)
        public static int Expansoes = 0;   
        // Quantidade de novos estados gerados
        public static int Geracoes = 0;    
        // Total de iterações de avaliação do ciclo principal
        public static int Avaliacoes = 0;  

        // Movimentos possíveis (cima, baixo, esquerda, direita)
        private static readonly (int dx, int dy)[] Movimentos =
        {
            (0, 1), (1, 0), (0, -1), (-1, 0)
        };

        // Retorna o custo de travessia da casa: se for normal (1), se for lenta (2)
        private static int CustoCasa(int valor)
        {
            return valor == 2 ? 2 : 1;
        }

        // Estima a satisfação potencial ainda possível
        private static int EstimarSatisfacaoRestante(int[,] mapa, HashSet<(int, int)> visitados, int tempoRestante)
        {
            // Tamanho do mapa (n x n)
            int n = mapa.GetLength(0); 
            // Lista de casas que ainda podem ser visitadas
            var visitaveis = new List<((int, int) pos, int custo, int pontos)>(); 

            // Percorre todas as posições do mapa
            for (int i = 0; i < n; i++)
            {
                for (int j = 0; j < n; j++)
                {
                    var pos = (i, j);
                    int val = mapa[i, j];
                    // Só casas não visitadas e acessíveis
                    if (!visitados.Contains(pos) && val != 10) 
                    {
                        // Custo para entrar na casa
                        int custo = CustoCasa(val); 
                        // Cada casa dá 1 ponto base
                        int pontos = 1; 
                        if (val < 0)
                            // Se for ponto de interesse, soma os pontos
                            pontos += Math.Abs(val); 
                        visitaveis.Add((pos, custo, pontos));
                    }
                }
            }

            // Ordena casas com mais pontos primeiro
            visitaveis.Sort((a, b) => b.pontos.CompareTo(a.pontos));
            int tempoUsado = 0;
            int estimativa = 0;

            // Soma pontos enquanto houver tempo para visitar
            foreach (var (_, custo, pontos) in visitaveis)
            {
                if (tempoUsado + custo <= tempoRestante)
                {
                    tempoUsado += custo;
                    estimativa += pontos;
                }
                else
                {
                    break;
                }
            }

            return estimativa;
        }

        // Algoritmo principal A* para encontrar o melhor passeio
        public static (List<(int, int)> caminho, int satisfacao)? PasseioSatisfacao(Instancia instancia)
        {
            // Inicialização dos contadores
            Expansoes = 0;
            Geracoes = 0;
            Avaliacoes = 0;

            var mapa = instancia.Mapa;
            int n = instancia.N;
            int tempoTotal = instancia.Tempo - 1;
            int meio = n / 2;

            // Define posições das 4 portas
            var portas = new List<(int, int)>
            {
                (0, meio), (n - 1, meio), (meio, 0), (meio, n - 1)
            };

            // Fila de prioridade A*
            var fila = new PriorityQueue<EstadoPasseio, int>(); 
            // Inicializa a melhor satisfação com o pior valor possível
            int melhorSatisfacao = int.MinValue; 
            // Caminho associado à melhor satisfação
            List<(int, int)> melhorCaminho = null; 

            var stopwatch = System.Diagnostics.Stopwatch.StartNew();
            // Limite de nós avaliados
            const int maxAvaliacoes = 1_000_000; 
            // Limite de tempo de execução
            const double tempoMaximoSegundos = 10.0; 

            // Enfileira todos os estados possíveis de entrada pelas portas
            foreach (var entrada in portas)
            {
                // Ignora porta se for parede (valor 10)
                if (mapa[entrada.Item1, entrada.Item2] == 10) continue; 
                // Custo de entrada (1 ou 2)
                int custoEntrada = CustoCasa(mapa[entrada.Item1, entrada.Item2]);
                int tempoRestanteInicial = tempoTotal - custoEntrada;
                // Marca a porta como visitada
                var visitados = new HashSet<(int, int)> { entrada };
                // Ganha 1 ponto satisfaçãoo por entrar
                int satisfacaoInicial = 1;
                // Soma pontos extra se for POI
                if (mapa[entrada.Item1, entrada.Item2] < 0)
                    satisfacaoInicial += Math.Abs(mapa[entrada.Item1, entrada.Item2]);
                // Heurística de satisfação futura
                int h = EstimarSatisfacaoRestante(mapa, visitados, tempoRestanteInicial);
                var inicial = new EstadoPasseio(entrada, tempoRestanteInicial, satisfacaoInicial, 0, visitados, new() { entrada }, h);
                // Enfileira com prioridade
                fila.Enqueue(inicial, -(inicial.Satisfacao + h - tempoRestanteInicial));
                // Conta esta geração inicial
                Geracoes++;
            }
            
            // Exploração da árvore de procura
            while (fila.Count > 0)
            {
                // Verifica limites de avaliação e tempo
                if (Avaliacoes++ > maxAvaliacoes || stopwatch.Elapsed.TotalSeconds > tempoMaximoSegundos)
                    break;
                // Retira o estado com maior prioridade da fila
                var estado = fila.Dequeue();
                // Conta esta expansão (estado processado) no total
                Expansoes++;

                // Gera vizinhos do estado atual
                foreach (var (dx, dy) in Movimentos)
                {
                    var vizinho = (estado.Posicao.Item1 + dx, estado.Posicao.Item2 + dy);
                    if (vizinho.Item1 < 0 || vizinho.Item2 < 0 || vizinho.Item1 >= n || vizinho.Item2 >= n) continue;
                    // Ignora paredes
                    if (mapa[vizinho.Item1, vizinho.Item2] == 10) continue;

                    int custo = CustoCasa(mapa[vizinho.Item1, vizinho.Item2]);
                    int novoTempo = estado.TempoRestante - custo;
                    // Se sem tempo, ignora
                    if (novoTempo < 0) continue; 

                    var novoCaminho = new List<(int, int)>(estado.Caminho) { vizinho };
                    var novosVisitados = new HashSet<(int, int)>(estado.Visitados);
                    int novaSatisfacao = estado.Satisfacao;
                    int novaPenalizacao = estado.PenalizacaoAtual;

                    // Se for uma casa nova, soma satisfação e reseta penalização
                    if (!novosVisitados.Contains(vizinho))
                    {
                        novaSatisfacao += 1;
                        if (mapa[vizinho.Item1, vizinho.Item2] < 0)
                            novaSatisfacao += Math.Abs(mapa[vizinho.Item1, vizinho.Item2]);
                        novaPenalizacao = 0;
                    }
                    else // Repetição de visita penaliza
                    {
                        novaPenalizacao++;
                        if (novaPenalizacao >= 2) novaSatisfacao--;
                    }

                    novosVisitados.Add(vizinho);

                    // Verifica se chegou a uma porta com tempo exatamente 0
                    if (portas.Contains(vizinho) && novoTempo == 0)
                    {
                        var caminhoFinal = new List<(int, int)>(novoCaminho);
                        int satisfacaoFinal = novaSatisfacao;
                        if (satisfacaoFinal > melhorSatisfacao)
                        {
                            melhorSatisfacao = satisfacaoFinal;
                            melhorCaminho = caminhoFinal;
                        }
                        continue;
                    }
                    // Calcula nova heurística (estimativa de satisfação ainda possível)
                    int heuristica = EstimarSatisfacaoRestante(mapa, novosVisitados, novoTempo);
                    
                    // Define a prioridade para a fila A* (prioridade mais alta = menor valor)
                    // A prioridade combina satisfação atual (com peso 3), heurística, e penaliza o cust
                    int prioridade = -((novaSatisfacao * 3) + heuristica - (custo * 2));
                    // Cria estado com a posição atualizada e valores acumulados
                    var novoEstado = new EstadoPasseio(vizinho, novoTempo, novaSatisfacao, novaPenalizacao, novosVisitados, novoCaminho, heuristica);
                    // Adiciona o novo estado à fila de prioridade
                    fila.Enqueue(novoEstado, prioridade);
                    // Conta como nova geração (nó gerado)
                    Geracoes++;
                }
            }
            // Retorna o melhor caminho encontrado (ou null se não houver)
            return melhorCaminho != null ? (melhorCaminho, melhorSatisfacao) : null;
        }
    }
}