class Player:
    def __init__(self, name):
        self.name = name  # Define o nome do jogador
        self.hand = []  # Inicializa a mão do jogador
        self.stayed = False  # Define se o jogador ficou
        self.folded = False  # Define se o jogador desistiu
        self.victory_percentage = 0.0  # Percentagem de vitórias inicial

    # Sobrecarga dos operadores de comparação
    def __lt__(self, other):
        return self.victory_percentage < other.victory_percentage

    def __gt__(self, other):
        return self.victory_percentage > other.victory_percentage

    def __eq__(self, other):
        return self.victory_percentage == other.victory_percentage

    def __repr__(self):
        return f"{self.name}: {self.victory_percentage}% vitórias"

    def add_card(self, card):
        self.hand.append(card)  # Adiciona uma carta à mão do jogador

    def clear_hand(self):
        self.hand = []  # Limpa a mão do jogador

    def calculate_hand_value(self, reveal_dealer=False):
        value = 0  # Inicializa o valor da mão
        aces = 0  # Conta os ases na mão

        if self.name == "Dealer" and not reveal_dealer:
            # Não inclui o valor da primeira carta do dealer
            for card in self.hand[1:]:  # Ignora a primeira carta, começa da segunda
                card_value = card.get_value()  # Pega o valor da carta
                value += card_value  # Adiciona ao valor total
                if card.value == 'Ás':  # Se for um Ás
                    aces += 1  # Conta os ases
        else:
            # Para qualquer jogador, calcula o valor normalmente
            for card in self.hand:
                card_value = card.get_value()  # Pega o valor da carta
                value += card_value  # Adiciona ao valor total
                if card.value == 'Ás':  # Se for um Ás
                    aces += 1  # Conta os ases

        for _ in range(aces):  # Ajusta o valor dos ases
            if value + 10 <= 21:
                value += 10  # Adiciona 10 ao valor se não ultrapassar 21

        return value  # Retorna o valor da mão

    def show_hand(self, reveal_dealer=False):
        if self.name == "Dealer" and not reveal_dealer:
            # Exibe a primeira carta como "####" e as restantes normalmente
            return f"#### {', '.join(str(card) for card in self.hand[1:])} - Valor: {self.calculate_hand_value(reveal_dealer)}"

        # Exibe todas as cartas do jogador
        hand_display = ", ".join(str(card) for card in self.hand)
        return f"{hand_display} - Valor: {self.calculate_hand_value(reveal_dealer)}"

    def take_action(self):
        action = ""  # Inicializa a ação
        while action not in ["P", "F", "D"]:
            action = input(f"{self.name}, escolha uma ação: (P)edir, (F)icar, (D)esistir: ").upper()  # Solicita a ação do jogador
        return action  # Retorna a ação escolhida

    def has_busted(self):
        return self.calculate_hand_value() > 21  # Verifica se o jogador estourou (ultrapassou 21)

    def has_stayed(self):
        return self.stayed  # Verifica se o jogador ficou (não pediu carta)

    def has_folded(self):
        return self.folded  # Verifica se o jogador desistiu

    def has_blackjack(self):
        from utils import BLACKJACK
        return self.calculate_hand_value() == BLACKJACK  # Verifica se o valor da mão é BlackJack

class JogadorHumano(Player):
    def __init__(self, name):
        super().__init__(name)  # Chama o construtor da classe base

    def take_action(self):
        # O jogador humano escolhe uma ação
        action = ""
        while action not in ["P", "F", "D"]:
            action = input(f"{self.name}, escolha uma ação: (P)edir, (F)icar, (D)esistir: ").upper()  # Solicita a ação do jogador
        return action

class Dealer(Player):
    def __init__(self, name="Dealer"):
        super().__init__(name)  # Chama o construtor da classe base

    def take_action(self, deck):
        from utils import DEALER_LIMIT
        # O dealer toma uma ação automática baseada na lógica do jogo
        dealer_value = self.calculate_hand_value(reveal_dealer=True)
        if dealer_value < DEALER_LIMIT:  # O dealer vai pedir carta se tiver menos que 17
            return "P"
        return "F"  # Caso contrário, o dealer ficará
