import matplotlib.pyplot as plt # Resposta 2a
import tkinter as tk
from player import Player
import json
import os


class Classificacoes:
    def __init__(self):
        self.file = "classificacoes.json"  # Caminho do ficheiro onde as classificações são guardadas
        self.classificacoes = self.carregar_classificacoes()  # Carrega as classificações ao iniciar

    def carregar_classificacoes(self):
        if os.path.exists(self.file):  # Verifica se o ficheiro existe
            with open(self.file, "r") as f:
                return json.load(f)  # Carrega as classificações do ficheiro
        return {}  # Retorna um dicionário vazio se não houver classificações

    def guardar_classificacoes(self):
        with open(self.file, "w") as f:
            json.dump(self.classificacoes, f, indent=4)  # Guarda as classificações no ficheiro

    def atualizar_classificacoes(self, nome_jogador, venceu):
        if nome_jogador not in self.classificacoes:  # Verifica se o jogador já existe nas classificações
            self.classificacoes[nome_jogador] = {
                "maos_jogadas": 0,  # Inicializa as estatísticas do jogador
                "maos_ganhas": 0,
                "percentagem_vitoria": 0.0
            }

        self.classificacoes[nome_jogador]["maos_jogadas"] += 1  # Incrementa o número de mãos jogadas
        if venceu:
            self.classificacoes[nome_jogador]["maos_ganhas"] += 1  # Incrementa as mãos ganhas se o jogador venceu

        total_jogadas = self.classificacoes[nome_jogador]["maos_jogadas"]  # Total de mãos jogadas
        total_ganhas = self.classificacoes[nome_jogador]["maos_ganhas"]  # Total de mãos ganhas
        self.classificacoes[nome_jogador]["percentagem_vitoria"] = round((total_ganhas / total_jogadas) * 100, 2)  # Calcula a percentagem de vitórias
        self.guardar_classificacoes()  # Guarda as classificações atualizadas

    def calcular_percentagem_media(self):
        total_vitoria = sum(stats["percentagem_vitoria"] for stats in self.classificacoes.values()) # Soma todas as percentagens de vitória
        total_jogadores = len(self.classificacoes) # Conta o número total de jogadores
        return round(total_vitoria / total_jogadores, 2) if total_jogadores > 0 else 0.0 # Calcula a média ou retorna 0.0 se não houver jogadores

    def criar_grafico(self):
        # Ordenar os jogadores por percentagem de vitória (Top 5)
        top5_jogadores = sorted(self.classificacoes.items(), key=lambda x: x[1]["percentagem_vitoria"], reverse=True)[
                         :5]

        # Extrair nomes e percentagens
        nomes = [jogador[0] for jogador in top5_jogadores]
        percentagens = [jogador[1]["percentagem_vitoria"] for jogador in top5_jogadores]

        # Criar gráfico
        plt.figure(figsize=(8, 5))
        plt.barh(nomes, percentagens, color='orange')
        plt.xlabel("Percentagem de Vitória")
        plt.ylabel("Jogadores")
        plt.title("Percentagem de Vitória TOP 5")
        plt.gca().invert_yaxis()  # Melhor jogador no topo

        # Exibir o gráfico
        plt.show()

    def mostrar_classificacoes(self, root):
        # Cria uma nova janela para exibir as classificações
        classificacoes_window = tk.Toplevel(root)
        classificacoes_window.title("Classificações")
        classificacoes_window.geometry("500x500")

        # Verifica se há classificações
        if not self.classificacoes:
            tk.Label(classificacoes_window, text="Ainda não há classificações.", font=("Helvetica", 12)).pack(pady=20)
        else:
            # Converte as classificações em objetos Player
            jogadores = []
            for nome, stats in self.classificacoes.items():
                jogador = Player(nome)
                jogador.victory_percentage = stats["percentagem_vitoria"]
                jogadores.append(jogador)

            # Ordena os jogadores usando a sobrecarga de operadores (decrescente)
            top10 = sorted(jogadores, key=lambda x: x.victory_percentage, reverse=True)[:10]

            # Widget de texto para exibir as classificações
            text_widget = tk.Text(classificacoes_window, wrap="word", font=("Courier", 10))
            text_widget.pack(padx=10, pady=10, fill="both", expand=True)

            # Exibe o Top 10 com mãos jogadas, ganhas e percentagem de vitórias
            text_widget.insert("end", "===== TOP 10 - Estatísticas =====\n")
            for i, jogador in enumerate(top10, start=1):
                stats = self.classificacoes[jogador.name]
                text_widget.insert(
                    "end",
                    f"{i}. {jogador.name}: {stats['maos_jogadas']} mãos jogadas, "
                    f"{stats['maos_ganhas']} mãos ganhas, {jogador.victory_percentage}% vitórias\n"
                )

            # Calcula a percentagem média de vitórias dos jogadores
            media_vitoria = self.calcular_percentagem_media()

            # Exibe o Top 10 com a comparação vs a média
            text_widget.insert("end", "\n===== TOP 10 - Comparação com a Média =====\n")
            for i, jogador in enumerate(top10, start=1):
                stats = self.classificacoes[jogador.name]
                comparacao = "acima" if jogador.victory_percentage > media_vitoria else "abaixo"
                text_widget.insert(
                    "end",
                    f"{i}. {jogador.name}: {jogador.victory_percentage}% vitórias "
                    f"comparado com a média de {media_vitoria}% ({comparacao}).\n"
                )

            # Torna o texto apenas de leitura
            text_widget.config(state="disabled")

        # Botão para fechar a janela
        tk.Button(classificacoes_window, text="Fechar", command=classificacoes_window.destroy).pack(pady=10)

        # Botão para gerar o gráfico
        tk.Button(classificacoes_window, text="Gráfico", command=self.criar_grafico).pack(pady=10)