import tkinter as tk

class Player:
    def __init__(self, name):
        self.name = name  # Define o nome do jogador
        self.hand = []  # Inicializa a mão do jogador
        self.stayed = False  # Define se o jogador ficou
        self.folded = False  # Define se o jogador desistiu
        self.victory_percentage = 0.0  # Percentagem de vitórias inicial

    # Sobrecarga dos operadores de comparação
    def __lt__(self, other):
        return self.victory_percentage < other.victory_percentage

    def __gt__(self, other):
        return self.victory_percentage > other.victory_percentage

    def __eq__(self, other):
        return self.victory_percentage == other.victory_percentage

    def __repr__(self):
        return f"{self.name}: {self.victory_percentage}% vitórias"

    def add_card(self, card):
        self.hand.append(card)  # Adiciona uma carta à mão do jogador

    def clear_hand(self):
        self.hand = []  # Limpa a mão do jogador

    def calculate_hand_value(self, reveal_dealer=False):
        value = 0  # Inicializa o valor da mão
        aces = 0  # Conta os ases na mão

        if self.name == "Dealer" and not reveal_dealer:
            # Não inclui o valor da primeira carta do dealer
            for card in self.hand[1:]:  # Ignora a primeira carta, começa da segunda
                card_value = card.get_value()  # Pega o valor da carta
                value += card_value  # Adiciona ao valor total
                if card.value == 'A':  # Se for um Ás
                    aces += 1  # Conta os ases
        else:
            # Para qualquer jogador, calcula o valor normalmente
            for card in self.hand:
                card_value = card.get_value()  # Pega o valor da carta
                value += card_value  # Adiciona ao valor total
                if card.value == 'A':  # Se for um Ás
                    aces += 1  # Conta os ases

        for _ in range(aces):  # Ajusta o valor dos ases
            if value + 10 <= 21:
                value += 10  # Adiciona 10 ao valor se não ultrapassar 21

        return value  # Retorna o valor da mão

    def show_hand(self, reveal_dealer=False):  # Resposta 1a
        def combine_cards_ascii(cards_ascii):
            combined = [""] * 9  # Cada carta tem 9 linhas
            for card_ascii in cards_ascii:
                for i, line in enumerate(card_ascii):
                    combined[i] += line + "  "  # Adiciona espaço entre as cartas
            return "\n".join(combined)

        if self.name == "Dealer" and not reveal_dealer: # Representação em ASCI da carta virada do dealer
            hidden_card = [
                "┌─────────┐",
                "│░░░░░░░░░│",
                "│░░░░░░░░░│",
                "│░░░░░░░░░│",
                "│░░░░░░░░░│",
                "│░░░░░░░░░│",
                "│░░░░░░░░░│",
                "│░░░░░░░░░│",
                "└─────────┘",
            ]
            visible_cards = [card.generate_ascii() for card in self.hand[1:]]  # Cartas visíveis
            cards_ascii = [hidden_card] + visible_cards  # Combina a carta oculta com as visíveis
            hand_ascii = combine_cards_ascii(cards_ascii)
            return f"{hand_ascii}\nValor: {self.calculate_hand_value(reveal_dealer)}"

        # Para outros jogadores ou o dealer quando revelado
        cards_ascii = [card.generate_ascii() for card in self.hand]
        hand_ascii = combine_cards_ascii(cards_ascii)
        return f"{hand_ascii}\nValor: {self.calculate_hand_value(reveal_dealer)}\n"

    def take_action(self, mostrar_mensagem, all_players): # Resposta 1a
        action_var = tk.StringVar()  # Variável para armazenar a escolha do jogador

        def definir_acao(acao):
            action_var.set(acao)  # Define a ação escolhida

        # Opções de ação para o jogador
        opcoes = {
            "Pedir": "P",
            "Ficar": "F",
            "Desistir": "D"
        }

        # Mensagem personalizada com o valor da mão do jogador
        mensagem = "Valor das mãos atuais:\n"
        for player in all_players:
            mensagem += f"{player.name}: {player.calculate_hand_value(reveal_dealer=False)}\n"
        mensagem += f"\n{self.name}, escolha uma ação:\n\n"
        # Chama a função mostrar_mensagem com as opções
        mostrar_mensagem(
            mensagem=mensagem,
            largura=400,
            altura=400,
            titulo=f"Ação de {self.name}",
            opcoes=opcoes,
            acao_callback=definir_acao
        )

        return action_var.get()  # Retorna a ação escolhida

    def has_busted(self):
        return self.calculate_hand_value() > 21  # Verifica se o jogador estourou (ultrapassou 21)

    def has_stayed(self):
        return self.stayed  # Verifica se o jogador ficou (não pediu carta)

    def has_folded(self):
        return self.folded  # Verifica se o jogador desistiu

    def has_blackjack(self):
        from utils import BLACKJACK
        return self.calculate_hand_value() == BLACKJACK  # Verifica se o valor da mão é BlackJack

    def exibir_nome(self):  # Resposta 1c
        return self.name

class JogadorHumano(Player):
    def __init__(self, name):
        super().__init__(name)  # Chama o construtor da classe base

class Dealer(Player):
    def __init__(self, name="Dealer"):
        super().__init__(name)  # Chama o construtor da classe base

    def exibir_nome(self):  # Resposta 1c
        return f"Dealer ({self.name})"

    def take_action(self, deck): # Resposta 1a
        from utils import DEALER_LIMIT
        # O dealer toma uma ação automática baseada na lógica do jogo
        dealer_value = self.calculate_hand_value(reveal_dealer=True)
        if dealer_value < DEALER_LIMIT:  # O dealer vai pedir carta se tiver menos que o limite da dificuldade
            return "P"
        return "F"  # Caso contrário, o dealer ficará

class BotPlayer(Player): # Classe que representa um jogador controlado por IA - # Resposta 1c

    def take_action(self):  # Resposta 1c
        return "P" if self.calculate_hand_value() < 17 else "F"
