import numpy as np
import matplotlib.pyplot as plt


# ============================================================
# OSCILADOR HARMÓNICO (SEM ATRITO)
#   Equação Diferencial (4):
#       d^2 x / dt^2 + k/m x = 0 <=> d^2 x / dt^2 = -k/m x , como w = sqrt(k/m) (5)
#       logo d^2 x / dt^2 = -w^2 x(t)
#   Na forma geral usada em métodos numéricos:
#       d^2 x / dt^2 = f(t, x, dx/dt) (15), definindo dx/dt = v
#   Obtemos que:
#       d^2 x / dt^2 = f(t, x, v) = -w^2 x(t)
#
# Vamos usar RK2 (Heun), na forma:
#   k1x = v_i
#   k1v = f(t_i, x_i, v_i)
#   k2x = v_i + k1v*h
#   k2v = f(t_i+h, x_i + k1x*h, v_i + k1v*h)
#   x_{i+1} = x_i + 1/2 (k1x + k2x) h
#   v_{i+1} = v_i + 1/2 (k1v + k2v) h
# ============================================================

# =========================
# Dados do problema
# =========================
w = 1.0         # w (rad/s)
x0 = 1.0        # x(0) em metros
v0 = 0.0        # v(0) em m/s
h = 0.2         # passo temporal (s)
t_max = 50.0    # tempo total (s) 


def f(_t, x, _v):
    """
    Função f(t, x, v) que dá a aceleração v'(t), é chamada na forma geral f(t, x, v)

    Como o oscilador harmónico ideal, não depende nem de t nem de v, pois:
        f(t, x, v) = -w^2 x
    """
    return -(w * w) * x


def x_analitico(t):
    """
    Solução analítica para comparar com o método numérico.

    Para x'' + w^2 x = 0, com x(0)=x0 e v(0)=v0:
        x(t) = x0 cos(wt) + (v0/w) sin(wt)

    No nosso caso: w=1, x0=1, v0=0 => x(t) = cos(t)
    """
    if w == 0:
        # Caso limite: w=0 => movimento uniforme
        return x0 + v0 * t
    return x0 * np.cos(w * t) + (v0 / w) * np.sin(w * t)


def rk2_heun_kvalues(t, x, v, h):
    """
    Executa 1 passo do método RK2 (Heun), devolve os coeficientes k e o novo estado (x_next, v_next).

    Tal que:
      - k1x é a inclinação inicial de x (ou seja, v no início do passo)
      - k1v é a inclinação inicial de v (ou seja, a aceleração no início)
      - k2x e k2v são as inclinações estimadas no "fim do passo"
        usando uma previsão tipo Euler para x e v.

    Depois, fazemos a média das inclinações (início e fim) para atualizar.
    """
    # --- k1 (inclinações no início do intervalo) ---
    k1x = v            # x' = v
    k1v = f(t, x, v)   # v' = -ω² x

    # --- k2 (inclinações no fim do intervalo, usando previsão Euler) ---
    # Previsão Euler:
    #   x_E = x + k1x*h
    #   v_E = v + k1v*h
    # Então:
    k2x = v + k1v * h
    k2v = f(t + h, x + k1x * h, v + k1v * h)

    # --- Atualização (média das inclinações) ---
    x_next = x + 0.5 * (k1x + k2x) * h
    v_next = v + 0.5 * (k1v + k2v) * h

    return k1x, k1v, k2x, k2v, x_next, v_next


def main():
    # ------------------------------------------------------------
    # 1) Construir vetor de tempos: t = 0, h, 2h, ..., t_max
    # ------------------------------------------------------------
    n = int(round(t_max / h))
    t = np.array([i * h for i in range(n + 1)], dtype=float)

    # ------------------------------------------------------------
    # 2) Vetores para guardar a solução numérica
    #    x[i] ~ posição em t[i]
    #    v[i] ~ velocidade em t[i]
    # ------------------------------------------------------------
    x = np.zeros(n + 1, dtype=float)
    v = np.zeros(n + 1, dtype=float)

    # Guardar também os k's (para a tabela)
    # Nota: Optou-se por na última linha (t=t_max) não há k's, porque não há passo seguinte.
    k1x = np.full(n + 1, np.nan, dtype=float)
    k1v = np.full(n + 1, np.nan, dtype=float)
    k2x = np.full(n + 1, np.nan, dtype=float)
    k2v = np.full(n + 1, np.nan, dtype=float)

    # Condições iniciais:
    x[0], v[0] = x0, v0

    # ------------------------------------------------------------
    # 3) INTEGRAÇÃO
    #    A cada passo i calculamos (x[i+1], v[i+1]) a partir de (x[i], v[i]).
    # ------------------------------------------------------------
    for i in range(n):
        k1x[i], k1v[i], k2x[i], k2v[i], x[i + 1], v[i + 1] = rk2_heun_kvalues(t[i], x[i], v[i], h)

    # ------------------------------------------------------------
    # 4) Solução analítica (para comparação)
    # ------------------------------------------------------------
    x_an = x_analitico(t)

    # ------------------------------------------------------------
    # 5) TABELA (com: t, x, v, k's e x analítico)
    # ------------------------------------------------------------
    print(f"{'t (s)':>6} | {'x (m)':>12} | {'v (m/s)':>12} | {'k1x':>12} | {'k1v':>12} | {'k2x':>12} | {'k2v':>12} | {'x analit. (m)':>14}")
    print("-" * (6+3+12+3+12+3+12+3+12+3+12+3+12+3+14))

    def fmt(val):
        return f"{val:12.9f}" if np.isfinite(val) else f"{'—':>12}"

    for i in range(n + 1):
        print(
            f"{t[i]:6.1f} | {x[i]:12.9f} | {v[i]:12.9f} | {fmt(k1x[i])} | {fmt(k1v[i])} | {fmt(k2x[i])} | {fmt(k2v[i])} | {x_an[i]:14.9f}"
        )

    # ------------------------------------------------------------
    # 6) Guardar tabela em CSV (Excel)
    # ------------------------------------------------------------
    data = np.column_stack((t, x, v, k1x, k1v, k2x, k2v, x_an))
    np.savetxt(
        "tabela_RK2_Heun_h0p2.csv",
        data,
        delimiter=",",
        header="t(s),x(m),v(m/s),k1x,k1v,k2x,k2v,x_analit(m)",
        comments="",
        fmt="%.6f"
    )

    # ------------------------------------------------------------
    # 7) GRÁFICO
    # ------------------------------------------------------------
    # Gráfico de x(t): numérico vs analítico
    plt.figure()
    plt.plot(t, x, "--", label=f"linha 1: x(t) RK2-Heun (h={h})")
    plt.plot(t, x_an, "--", label="linha 2: x(t) analítico")
    plt.xlabel("t (s)")
    plt.grid(True)
    plt.legend()
    plt.title("Oscilador Harmónico — x(t)")

    plt.show()


if __name__ == "__main__":
    main()